
/*
matrixmangle.cpp

matrix/coordinate system conversion methods from MAX to Renderware should be migrated here
*/

#include "stdafx.h"
#include "DffExp.h"
#include "characterstudio.h"
#include "decomp.h"

const Matrix3
ToRenderwareAxisSystem( const Matrix3 &theMatrix, INode *node )
//Convert the given matrix to Renderware's axis system,
//relative to the given node in MAX's scene heirarcy
//(Currently, for correcting Character Studio's silly axis system
//to be consistent with other objects in MAX,
//but who knows what other sillyness Kinetix has up their sleeve)
{    
    if (node != 0)
    {
        char *name = node->GetName();

        //Only correct axis system if biped is on top of hierachy.
        //If biped happens to be a child of some other object, do nothing,
        //and everything just works out fine. (yes, totally weird ain't it?)
        if ((node->GetParentNode() != 0)
            && node->GetParentNode()->IsRootNode()
            && IsABipedNode( node ) )
        {
            return Matrix3( theMatrix ) *= Matrix3( Point3(0,0,1),
                                                    Point3(1,0,0),
                                                    Point3(0,1,0),
                                                    Point3(0,0,0) );
        }
    }

    return theMatrix;
}

Matrix3
WorldSpaceMatrix()
{
    Point3 row(0.0f, 0.0f, -1.0f);
    Matrix3 worldSpace;
    worldSpace.IdentityMatrix();
    worldSpace.SetRow(1, row);
    row.y = 1.0f;
    row.z = 0.0f;
    worldSpace.SetRow(2, row);
    return (worldSpace);
}

RwMatrix *
Matrix3ToRwMatrix(Matrix3 maxMat, RwMatrix *rwMatrix)
//Copies contents of a MAX matrix to a Renderware Matrix verbatim
//(This fn doesn't need to be a member)
{
    Point3 row;

    /* Direct matrix conversion no axis system adjustment etc */
    row = maxMat.GetRow(0);
    memcpy(&(rwMatrix->right), &row, sizeof(Point3));
    row = maxMat.GetRow(1);
    memcpy(&(rwMatrix->up), &row, sizeof(Point3));
    row = maxMat.GetRow(2);
    memcpy(&(rwMatrix->at), &row, sizeof(Point3));
    row = maxMat.GetRow(3);
    memcpy(&(rwMatrix->pos), &row, sizeof(Point3));

    RwMatrixUpdate(rwMatrix);

    return(rwMatrix);
}

Matrix3
DFFExport::GetObjMatrix(INode *node, RwInt32 frameNum, RwBool clearRoot)
{
    /* get node's LTM */

    //This method returns the world space transformation matrix of the node at the specified time.
    //This matrix contains its parents transformation.
    //This matrix does not include the object-offset transformation,
    //or any world space modifier (Space Warp) affects.
    //If you select a single node and change the reference coordinate system to 'Local',
    //you'll see the node's axes tripod displayed.
    Matrix3 nodeTM = node->GetNodeTM(frameNum);

    //this method returns the transformation matrix the object needs to be multiplied by to transform it into world space.
    //At times, this matrix may be the identity.  For example, a deformable object that has a Space Warp applied to it
    //would already have been translated into world space by the space warp.  Since the object is already in world space
    //the matrix needed to get it there is the identity.
    //This matrix would not be the identity for a deformable object with only object space modifiers applied.
    //This object would indeed need to be transformed.  In this case the TM returned would include the NodeTM plus the
    //object-offset transformation.  So, GetObjectTM() is dependent on the context when it is called -- it will either
    //be equal to GetObjectTMAfterWSM() or GetObjectTMBeforeWSM().  Developers should use GetObjectTMBeforeWSM() if what
    //is wanted is the object TM and not the identity matrix.
    //For non-deformable objects this matrix may include the NodeTM, the object-offset transformation and the world
    //space modifier affect.
    //This matrix could be used, for example, if you have a TriObject and wanted to get the world space coordinate of
    //one of its vertices.  You could do this by taking the vertex coordinate in object space and multiplying it by the
    //matrix returned from this method.
    Matrix3 objectTM = node->GetObjectTM(frameNum);

    /* The object offset including the nodeTM scaling
       is equivalent to objectTM * Inverse(nodeTMWithoutScaling) */
    nodeTM.NoScale();
    m_objectOffset = objectTM * Inverse(nodeTM);

    /* Refetch the nodeTM with scaling */
    nodeTM = node->GetNodeTM(frameNum);

    if (!node->GetParentNode()->IsRootNode())   //i.e. this node isn't the top of a heirarchy
    {
        INode* parentNode = node->GetParentNode();
        Matrix3 parentNodeTM = parentNode->GetNodeTM(frameNum);

        parentNodeTM.NoScale();
        /* Make the nodeTM a modelling matrix but leave in
           parent scaling for the translation. We'll blow
           away all scaling soon anyway */
        nodeTM = nodeTM * Inverse(parentNodeTM);
    }
    else                                       //i.e. this node is the top of a heirarchy
    {
        if (m_retainObjInWorldSpace)
        {
            nodeTM = nodeTM * WorldSpaceMatrix();
        }
        else
        {
            if (clearRoot)
            {
                /* take out rotations and translations from root */
                nodeTM.IdentityMatrix();
            }
        }
    }

    /* Remove scaling from the nodeTM */
    nodeTM.NoScale();

    nodeTM.SetRow(3, nodeTM.GetRow(3) * m_nScaleFactor);

    return (nodeTM);
}

Matrix3
DFFExport::GetNodeLTM(INode *node)
{            
    if (node->GetParentNode() && !node->GetParentNode()->IsRootNode())
    {        
        Matrix3 parentLTM = GetNodeLTM(node->GetParentNode());
        
        Matrix3 objMat = GetObjMatrix(node, m_tvStart, TRUE);

        objMat = objMat * parentLTM;
        
        return objMat;
    }
    else
    {
        Matrix3 objMat = GetObjMatrix(node, m_tvStart, TRUE);
        //Character studio has a different idea of what is "up" than the rest of MAX.
        //Reorient this matrix to have the same "up" as everything else
        objMat = ToRenderwareAxisSystem( objMat, node );
        return objMat;
    }    
}

void NegativeScaleCheck( INode *node, TimeValue when, WarningList &warningList )
//test if node's matrix has a negative scaling applied
// & generate a warning message if it does.
{
    Matrix3 objMat = node->GetNodeTM(when);

    Point3 r0=objMat.GetRow(0);
    Point3 r1=objMat.GetRow(1);
    Point3 r2=objMat.GetRow(2);
    Point3 r3=objMat.GetRow(3);
    float Det;

    Det=r0.x*(r1.y*r2.z-r1.z*r2.y)+r1.x*(r2.y*r0.z-r0.y*r2.z)+r2.x*(r0.y*r1.z-r1.y*r0.z);

    char temp[2048];
    sprintf(temp,"TM=( (%.3f %.3f %.3f) (%.3f %.3f %.3f) (%.3f %.3f %.3f) (%.3f %.3f %.3f) ) Det=%.5f",
        r0.x,r0.y,r0.z,r1.x,r1.y,r1.z,r2.x,r2.y,r2.z,r3.x,r3.y,r3.z,Det);
    warningList.add(CWarning(wtInformational,node->GetName(),temp));

    if (Det<0)
    {
        warningList.add(CWarning(wtWarning,node->GetName(),_T("Negative scaling detected.")));
    }

    if ((Det<=c_SingularMatrixDeterminant)&&(Det>=-c_SingularMatrixDeterminant))
    {
        warningList.add(CWarning(wtError,node->GetName(),_T("Singular transformation matrix detected.")));
    }
}

Matrix3 
Uniform_Matrix(Matrix3 orig_cur_mat)
//take a matrix and return only translation/rotation parts in a new matrix
//(various scalings, including negative ones, will be removed).
{          
    AffineParts   parts;  
    Matrix3       mat;   
     
    //Remove  scaling  from orig_cur_mat
    //1) Decompose original and get decomposition info
    decomp_affine(orig_cur_mat, &parts); 
     
    //2) construct 3x3 rotation from quaternion parts.q
    parts.q.MakeMatrix(mat);
     
    //3) construct position row from translation parts.t  
    mat.SetRow(3,  parts.t);
     
    return(mat);
}
